/*
* Creation date : Fri Mar 02 10:45:20 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* CE2_HMAC() function. 
*
* \version CE2_HMAC.c#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_KDF.h"
#include "LLF_HMAC.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  CE2_HMAC
*
* Inputs:
* @param OperationMode [in] - The operation mode:
*                  SHA1, SHA224, SHA256, SHA384, SHA512, MD5;
* @param key_ptr [in] - The pointer to the user's key buffer;
* @param keySize [in] - The size of the received key;
* @param DataIn_ptr [in] - A pointer to the buffer that stores the data to be hashed;
* @param DataSize [in] - The size of the data to be hashed, in bytes;
* @param HmacResultBuff [out] - A pointer to the target buffer where the 
*                   HMAC result stored in the context is loaded to;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*   - CE2_HMAC_ILLEGAL_HASH_OP_MODE_ERROR
*   - CE2_HMAC_INVALID_KEY_PTR_ERROR
*   - CE2_HMAC_INVALID_DATA_IN_PTR_ERROR
*   - CE2_HMAC_INVALID_HASH_RESULT_BUFF_PTR_ERROR
*
* \brief \b
* Description:
*   The function allocates an internal HASH Context, it initializes the 
* HASH Context with the cryptographic attributes that are needed for 
* the HASH block operation (initializes H's value for the HASH algorithm).
* Next the function loads the engine with the initializing values, 
* and then processes the data block, calculating the hash.
* Finally, the function return the message digest of the data buffer .
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_HMAC
*    to calculate HASH of inputted data with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_HMAC  (
                         CE2_HASH_OperationMode_t  OperationMode,	/* in */
                         DxUint8_t                 *key_ptr,		/* in */
                         DxUint16_t                keySize,			/* in */
                         DxUint8_t                 *DataIn_ptr,		/* in */
                         DxUint32_t                DataSize,		/* in */
                         CE2_HASH_Result_t         HmacResultBuff )/* out */
{
	if (OperationMode < 0 || OperationMode >= CE2_HASH_NumOfModes)
		return CE2_HMAC_ILLEGAL_HASH_OP_MODE_ERROR;

	if (key_ptr == DX_NULL)
		return CE2_HMAC_INVALID_KEY_PTR_ERROR;

	if (DataIn_ptr == DX_NULL && DataSize != 0)
		return CE2_HMAC_INVALID_DATA_IN_PTR_ERROR;

	if (HmacResultBuff == DX_NULL)
		return CE2_HMAC_INVALID_HASH_RESULT_BUFF_PTR_ERROR;

	return LLF_HMAC(OperationMode,key_ptr,keySize,
					DataIn_ptr,DataSize,HmacResultBuff);
} /* End of CE2_HMAC */
